#!/bin/sh

echo "+++++ Checking current user rights..."

if [ "$(id -u root)" = "$(id -u)" ] ; then
    echo "Passed..."
else
    echo "ERROR: The current user id is not root (`id|awk '{print $1}'`)."
    exit 1
fi
echo


if [ "X$ACTIANZEN_ROOT" = "X" ] ; then
    ACTIANZEN_ROOT=/usr/local/actianzen
fi
export ACTIANZEN_ROOT

LIBDIR=lib

PATH=$ACTIANZEN_ROOT/bin:$PATH

if [ $(uname -s) = "Darwin" ]
then
    DYLD_LIBRARY_PATH=$ACTIANZEN_ROOT/bin:$ACTIANZEN_ROOT/lib:$ACTIANZEN_ROOT/lib64:$DYLD_LIBRARY_PATH
    export DYLD_LIBRARY_PATH
    SO_EXTENSION="dylib"
    SU_ARGS="-l"
else
    LD_LIBRARY_PATH=$ACTIANZEN_ROOT/bin:$ACTIANZEN_ROOT/lib:$ACTIANZEN_ROOT/lib64:$LD_LIBRARY_PATH
    export LD_LIBRARY_PATH
    SO_EXTENSION="so"
    SU_ARGS=
fi

# PSQL-6812 - v14 or later incompatible with v13 and earlier.
echo "+++++ Checking for previous installs..."
if [ -f /usr/local/psql/lib64/libpsqlmif.${SO_EXTENSION} ] ; then
    echo "ERROR: An incompatible version of PSQL is currently installed.  Please uninstall the incompatible version before installing Zen."
    exit 1
fi
echo "Passed..."
echo

# PSQL-6981 - Cannot install v14 or later if legacy username is running any process.
echo "+++++ Checking for running processes owned by user 'psql'..."
PROC_LIST_HEADERS=`ps -ef | head -n1`
LEGACY_OWNER_PROCS=`ps -fu psql 2>/dev/null | grep psql | grep -v grep`
if [ "X$LEGACY_OWNER_PROCS" != "X" ] ; then
    echo "ERROR: One or more processes owned by user 'psql' is currently running. Please stop all processes owned by user 'psql' prior to installing Zen."
    echo "Running processes owned by user 'psql':"
    echo "$PROC_LIST_HEADERS"
    echo "$LEGACY_OWNER_PROCS"
    echo
    exit 1
fi
echo "Passed..."
echo

if [ $(uname -s) != "Darwin" ]
then
    echo "+++++ Checking for system dependencies..."

    # PSQL-8641: Remove use of 'file -L'. Makefile updates placeholder tags
    # dynamically at build time.
    if [ ! -e "/lib/ld-linux.so.2" ] ; then
        echo "ERROR: /lib/ld-linux.so.2 not found. This package requires this 32-bit version of ld-linux."
        exit 1
    fi
fi

if [ $(uname -s) != "Darwin" ]; then

    check_deplib() {
        LIB_FILE=$1              # name of lib to check.
        ZEN_FILE_TO_CHECK=$2     # binary to use for check e.g. $ACTIANZEN_ROOT/bin/psregsvr
        SYS_INSTALLED_LIB_PATH=`ldd $ZEN_FILE_TO_CHECK | grep $LIB_FILE | cut -d " " -f 3`
        if [ ! -f "$SYS_INSTALLED_LIB_PATH" ]; then
            echo "ERROR: $LIB_FILE not found. This package requires this library."
            exit 1
        fi
    }

    for FILE in $ACTIANZEN_ROOT/$LIBDIR/libcsi100.${SO_EXTENSION}.[0-9]*
    do
        if [ \( -f $FILE \) -a \( ! -h $FILE \) ]; then
            CHECK_LIB=$FILE
            break
        fi
    done
    if [ "$CHECK_LIB" ]; then
        check_deplib libcrypt.so.1 $CHECK_LIB
    fi

fi # not Darwin

echo "Passed..."
echo

# Done with checks. Begin using installed files

# Determine if system libs (libstdc++, libgcc_s) meets Zen version requirements and remove
# bundled version if it does.
if [ $(uname -s) != "Darwin" ]; then

    # Function: check_lib(LIB_PATH, ZEN_REQUIRED_LIB_VER, ZEN_FILE_TO_CHECK)
    # requires: readelf command
    check_lib() {
        LIB_PATH=$1              # path of lib to check relative to $ACTIANZEN_ROOT
                                 # e.g. lib64/libstdc++.so.6
        ZEN_REQUIRED_LIB_VER=$2  # version tag e.g GLIBCXX_N.N.N
        ZEN_FILE_TO_CHECK=$3     # binary to use for check e.g. $ACTIANZEN_ROOT/bin/psregsvr

        LIB_FILE=`basename $LIB_PATH`         # e.g. libsdtc++.so.6
        LIB_DIR_REL=`dirname $1`              # e.g. lib or lib64
        LIB_DIR=$ACTIANZEN_ROOT/$LIB_DIR_REL  # e.g. /usr/local/actianzen/lib64

        echo "+++++ Checking version support for file $LIB_PATH ..."
        # Verify readelf is available.
        # Note: It is assumed that caller will warn if readelf is not available.
        READELF=`which readelf`
        if [ -f "$READELF" ]; then
            # First rename the bundled lib in case $ACTIANZEN_ROOT is
            # already in the path so the ldd check below will locate the system
            # library instead of the Zen-provided library (or not find it at all).
            if [ -f $LIB_DIR/$LIB_FILE ]; then
                mv $LIB_DIR/$LIB_FILE $LIB_DIR/$LIB_FILE.zen
            fi

            # Now locate system library path from file dependencies for $ZEN_FILE_TO_CHECK.
            if [ -f $ZEN_FILE_TO_CHECK ]; then
                echo "Obtaining $LIB_PATH dependencies from $ZEN_FILE_TO_CHECK"
                SYS_INSTALLED_LIB_PATH=`ldd $ZEN_FILE_TO_CHECK | grep $LIB_FILE | cut -d " " -f 3`
                if [ -f "$SYS_INSTALLED_LIB_PATH" ]; then
                    echo "Checking for required version info in $SYS_INSTALLED_LIB_PATH"
                    FOUND_ZEN_LIB_DEP=`readelf -V $SYS_INSTALLED_LIB_PATH | grep $ZEN_REQUIRED_LIB_VER`
                    if [ "X$FOUND_ZEN_LIB_DEP" = "X" ]; then
                        echo "Required version support not found, Zen will use $LIB_FILE from install."
                        # Revert previous rename of libraries bundled with Zen install.
                        if [ -f $LIB_DIR/$LIB_FILE.zen ]; then
                            # Revert previous rename of libraries bundled with Zen install.
                            mv $LIB_DIR/$LIB_FILE.zen $LIB_DIR/$LIB_FILE
                        fi
                    else
                        echo "Required version support found, Zen will use $SYS_INSTALLED_LIB_PATH"
                        # Must also remove previously renamed file or it will get a symlink created
                        # for it later in this script.
                        if [ -f $LIB_DIR/$LIB_FILE.zen ]; then
                            rm -f $LIB_DIR/$LIB_FILE.zen
                        fi
                    fi
                else
                    echo "Required version support not found, Zen will use $LIB_FILE from install."
                    # Revert previous rename of libraries bundled with Zen install.
                    if [ -f $LIB_DIR/$LIB_FILE.zen ]; then
                        # Revert previous rename of libraries bundled with Zen install.
                        mv $LIB_DIR/$LIB_FILE.zen $LIB_DIR/$LIB_FILE
                    fi
                fi
            else
                echo "The file $ZEN_FILE_TO_CHECK is not present in the Zen install package, skipping dependency check."
            fi

        fi
    }

    # Verify readelf is available.
    READELF=`which readelf`
    if [ ! -f "$READELF" ]; then
        echo "WARNING: Unable to determine versioning information of system libraries"
        echo "libstdc++.so.6 and libgcc_s.so.1 (missing readelf command)."
        echo "Libraries bundled with the Zen installation will be used."
        echo "These may not be compatible with other utilities."
    else
        check_lib "lib/libstdc++.so.6" "GLIBCXX_3.4.21" "$ACTIANZEN_ROOT/bin/psregsvr"
        check_lib "lib/libgcc_s.so.1" "GCC_7.0.0" "$ACTIANZEN_ROOT/bin/psregsvr"
    fi
fi # not Darwin

echo

USERNAME=zen-svc
GROUPNAME=zen-data
USERSHL=/bin/bash
USERSHLPROFILE1=.bash_profile
USERSHLPROFILE2=.bashrc
USERSHLPROFILE3=.cshrc

echo "Updating user $USERNAME environment..."
# Update the library path in the zen-svc profile files to include 32-bit lib folder path.
if [ $(uname -s) = "Darwin" ]
then
    HOMEDIR=/Users/"$USERNAME"
else
    HOMEDIR=`grep "^$USERNAME:" /etc/passwd | cut -d: -f6`
fi

PROFILE1=$HOMEDIR/$USERSHLPROFILE1
PROFILE2=$HOMEDIR/$USERSHLPROFILE2
PROFILE3=$HOMEDIR/$USERSHLPROFILE3

if [ $(uname -s) = "Darwin" ]
then
    echo export DYLD_LIBRARY_PATH=\$ACTIANZEN_ROOT/lib:\$DYLD_LIBRARY_PATH >> $PROFILE1
    echo export DYLD_LIBRARY_PATH=\$ACTIANZEN_ROOT/lib:\$DYLD_LIBRARY_PATH >> $PROFILE2
    echo setenv DYLD_LIBRARY_PATH \$ACTIANZEN_ROOT/lib:\$DYLD_LIBRARY_PATH >> $PROFILE3
else
    echo export LD_LIBRARY_PATH=\$ACTIANZEN_ROOT/lib:\$LD_LIBRARY_PATH >> $PROFILE1
    echo export LD_LIBRARY_PATH=\$ACTIANZEN_ROOT/lib:\$LD_LIBRARY_PATH >> $PROFILE2
    echo setenv LD_LIBRARY_PATH=\$ACTIANZEN_ROOT/lib:\$LD_LIBRARY_PATH >> $PROFILE3
fi
echo

echo "+++++ Setting up symbolic links for libraries..."

# These used to be in the RPM spec file, but we moved them
# here for tarball installs.
rm -f $ACTIANZEN_ROOT/lib/libbtrvif.${SO_EXTENSION}

LDCONFIG=/sbin/ldconfig
if [ ! -x $LDCONFIG ] ; then
    LDCONFIG=/usr/sbin/ldconfig
fi

for FILE in $ACTIANZEN_ROOT/$LIBDIR/libpsqlmif.${SO_EXTENSION}.[0-9]*
do
    if [ \( -f $FILE \) -a \( ! -h $FILE \) ]; then
        MIF_FILE=$FILE
        break
    fi
done
if [ "$MIF_FILE" ]; then
    # ..../libpsqlmif.so.15.10.NNN.MMM --> 15
    # ..../libpsqlmif.so.15 -> 15
    PSQL_VER=`echo $MIF_FILE | sed -e 's/^.*\.'"${SO_EXTENSION}"'\.\([0-9]*\).*$/\1/'`
else
    if [ $(uname -s) = "Darwin" ]; then
        # ..../libpsqlmif.15.dylib --> 15
        for FILE in $ACTIANZEN_ROOT/$LIBDIR/libpsqlmif.[0-9]*.dylib
        do
            if [ \( -f $FILE \) -a \( ! -h $FILE \) ]; then
                MIF_FILE=$FILE
                break
            fi
        done
        PSQL_VER=`echo $MIF_FILE | sed -e 's/^.*\.\([0-9]*\)\.dylib$/\1/'`
    fi
fi
echo "++++ MIF_FILE: $MIF_FILE (PSQL_VER=$PSQL_VER)"
PS_VER=$PSQL_VER

if [ ! $(uname -s) = "Darwin" ]
then
    # Use ldconfig to create links to major versions
    # selinux: ldconfig runs in its own domain and may not have
    # permissions to write to tty.
    $LDCONFIG -n $ACTIANZEN_ROOT/lib 2>&1 | cat

    dso_symlink () {
        dso=$1
        ver=$2
        # libmkc3.so.15.10.NNN.000 <-- libmkc3.so.15 [ldconfig for compmake]
        # libmkc3.so.15 <-- libmkc3.so [compmake; cmake]
        if [ -f $ACTIANZEN_ROOT/$LIBDIR/$dso.$ver ]; then
            (cd $ACTIANZEN_ROOT/$LIBDIR; ln -sf $dso.$ver $dso )
        fi
    }
else

    dso_symlink () {
      dso=$1
      ver=$2
      # libmkc3.dylib.15.10.NNN.000 <-- libmkc3.dylib.15 [compmake]
      # libmkc3.dylib.15 <-- libmkc3.dylib [compmake]
      if [ -f $ACTIANZEN_ROOT/$LIBDIR/$dso.$ver.[0-9]*.* ]; then
          (cd $ACTIANZEN_ROOT/$LIBDIR;
           ln -sf $dso.$ver.[0-9]*.* $dso.$ver;
           ln -sf $dso.$ver $dso)
      fi
      dso=`basename $1 .${SO_EXTENSION}`
      # libmkc3.15.dylib <-- libmkc3.dylib [cmake]
      if [ -f $ACTIANZEN_ROOT/$LIBDIR/$dso.$ver.${SO_EXTENSION} ]; then
          (cd $ACTIANZEN_ROOT/$LIBDIR;
           ln -sf $dso.$ver.${SO_EXTENSION} $dso.${SO_EXTENSION})
      fi
    }

fi
dso_symlink libclientrb.${SO_EXTENSION} $PSQL_VER
dso_symlink libcsi100.${SO_EXTENSION} $PSQL_VER
dso_symlink libdbcsipxy.${SO_EXTENSION} $PSQL_VER
dso_symlink libdcm100.${SO_EXTENSION} $PSQL_VER
dso_symlink libenginelm.${SO_EXTENSION} $PSQL_VER
dso_symlink libexp010.${SO_EXTENSION} $PSQL_VER
dso_symlink liblegacylm.${SO_EXTENSION} $PSQL_VER
dso_symlink liblicmgrrb.${SO_EXTENSION} $PSQL_VER
dso_symlink libmkc3.${SO_EXTENSION} $PSQL_VER
dso_symlink libmkderb.${SO_EXTENSION} $PSQL_VER
dso_symlink libodbccr.${SO_EXTENSION} 1
dso_symlink libpceurop.${SO_EXTENSION} $PS_VER
dso_symlink libpctlgrb.${SO_EXTENSION} $PSQL_VER
dso_symlink libpscl.${SO_EXTENSION} $PS_VER
dso_symlink libpscore.${SO_EXTENSION} $PS_VER
dso_symlink libpscp932.${SO_EXTENSION} $PS_VER
dso_symlink libpsdom.${SO_EXTENSION} $PS_VER
dso_symlink libpseucjp.${SO_EXTENSION} $PS_VER
dso_symlink libpssax.${SO_EXTENSION} $PS_VER
dso_symlink libpsutilrb.${SO_EXTENSION} $PS_VER
dso_symlink libpvmsgrb.${SO_EXTENSION} $PSQL_VER
dso_symlink libsrderb.${SO_EXTENSION} $PSQL_VER
dso_symlink libupiapirb.${SO_EXTENSION} $PSQL_VER
dso_symlink libxlate.${SO_EXTENSION} $PSQL_VER

cd $ACTIANZEN_ROOT/$LIBDIR
if [ -f $ACTIANZEN_ROOT/$LIBDIR/libodbc.${SO_EXTENSION}.1*0 ]; then
    if [ ! -f $ACTIANZEN_ROOT/$LIBDIR/libodbc.${SO_EXTENSION}.1 ]; then
        ln -sf libodbc.${SO_EXTENSION}.1*0 libodbc.${SO_EXTENSION}.1
    fi
elif [ -f $ACTIANZEN_ROOT/$LIBDIR/libodbc.1.0.dylib ]; then
    if [ ! -f $ACTIANZEN_ROOT/$LIBDIR/libodbc.1.dylib ]; then
        ln -sf libodbc.1.0.${SO_EXTENSION} libodbc.1.dylib
    fi
fi
cd $ACTIANZEN_ROOT
dso_symlink libodbc.${SO_EXTENSION} 1

dso_symlink libpsqldti.${SO_EXTENSION} $PSQL_VER
dso_symlink libclientlm.${SO_EXTENSION} $PSQL_VER
dso_symlink libbtrieveC.${SO_EXTENSION} $PSQL_VER
dso_symlink libbtrieveCpp.${SO_EXTENSION} $PSQL_VER
dso_symlink libpsqlmif.${SO_EXTENSION} $PSQL_VER
dso_symlink libpsqlnsl.${SO_EXTENSION} $PSQL_VER
dso_symlink libpsxdebug.${SO_EXTENSION} $PSQL_VER
dso_symlink libpsqlcsm.${SO_EXTENSION} $PSQL_VER
dso_symlink libpsqlcsp.${SO_EXTENSION} $PSQL_VER
dso_symlink libodbcci.${SO_EXTENSION} $PSQL_VER
dso_symlink libcsi100.${SO_EXTENSION} $PSQL_VER
dso_symlink libpsqlmpm.${SO_EXTENSION} $PSQL_VER

dso_symlink libels.${SO_EXTENSION} $PSQL_VER

echo

SELINUX_ENABLED=/usr/sbin/selinuxenabled

if [ -x $SELINUX_ENABLED ] && $SELINUX_ENABLED; then
    /sbin/restorecon -R $ACTIANZEN_ROOT
    if [ -d "$ACTIANZEN_ROOT"/lib ]
    then
    /usr/bin/chcon -t textrel_shlib_t "$ACTIANZEN_ROOT"/lib/*.so "$ACTIANZEN_ROOT"/lib/*.so.*
    fi
fi


echo "+++++ Registering the PCOM libraries..."

PSREGSVR=psregsvr
# Sym links must already exist
dso_register () {
    dso=$1
    if [ -f $ACTIANZEN_ROOT/$LIBDIR/$dso ]; then
        $ACTIANZEN_ROOT/bin/$PSREGSVR $ACTIANZEN_ROOT/$LIBDIR/$dso
        if [ $? -ne 0 ]; then
            echo "ERROR: PCOM registration failed for $dso."
            exit 1;
        fi
    fi
}
dso_register libpctlgrb.${SO_EXTENSION}
dso_register libpceurop.${SO_EXTENSION}
dso_register libpssax.${SO_EXTENSION}
dso_register libpsdom.${SO_EXTENSION}
dso_register libpsutilrb.${SO_EXTENSION}
dso_register libclientrb.${SO_EXTENSION}
dso_register libupiapirb.${SO_EXTENSION}
dso_register libcsi100.${SO_EXTENSION}
dso_register libmkc3.${SO_EXTENSION}
dso_register libpvmsgrb.${SO_EXTENSION}
dso_register libpsqlcsm.${SO_EXTENSION}
dso_register libpsqlcsp.${SO_EXTENSION}
dso_register libdbcsipxy.${SO_EXTENSION}
dso_register libpsqlmpm.${SO_EXTENSION}
dso_register libpscp932.${SO_EXTENSION}
dso_register libsrderb.${SO_EXTENSION}


echo "+++++ Making user $USERNAME owner of all files in $ACTIANZEN_ROOT/lib and $ACTIANZEN_ROOT/bin..."
chown -f $USERNAME:$GROUPNAME $ACTIANZEN_ROOT
chown -Rf $USERNAME:$GROUPNAME $ACTIANZEN_ROOT/lib
chown -Rf $USERNAME:$GROUPNAME $ACTIANZEN_ROOT/bin
chown -f $USERNAME:$GROUPNAME $ACTIANZEN_ROOT/etc

echo

echo "+++++ Install has successfully completed."
echo
