/**
 * This system procedure is used to rename objects.
 *
 * Arguments
 *   object_name - the current name of the user object (table, view, column,
 *   stored procedure, user defined function, trigger, or index. If the object
 *   to be renamed is a column in a table, object_name must be in the form
 *   table.column. If the object to be renamed is an index, object_name must be
 *   in the form table.index. If the object to be renamed is a trigger,
 *   object_name must be in the form table.trigger. If the object to be renamed
 *   is a table, object_name must be in the form table. If the object to be renamed
 *   is a view, object_name must be in the form view. If the object to be renamed
 *   is a procedure, object_name must be in the form procedure. If the object to
 *   be renamed is a function, object_name must be in the form function.
 *   
 *   new_name - is the new name for the specified object. new_name must be a one-part
 *   name and must follow the rules for identifiers.
 *   
 *   object_type - is the type of object being renamed. object_type is varchar(13)
 *                 and can be one of these values:
 *                 COLUMN
 *                 INDEX
 *                 TRIGGER
 *                 TABLE
 *                 VIEW
 *                 PROCEDURE
 *                 FUNCTION
 */
CREATE PROCEDURE psp_rename
(
    in :object_name VARCHAR(776),   -- original object name
    in :new_name VARCHAR(776),      -- new object name
    in :object_type VARCHAR(13)     -- object type
)
as
BEGIN
    -- Variable to hold the dynamic query 
    DECLARE :stmt_psp_rename VARCHAR(1552);

    -- variable to keep table name
    DECLARE :table_name VARCHAR(776);

    -- variable to keep column name
    DECLARE :column_name VARCHAR(776);

    -- variable to keep table nam
    DECLARE :dot_position INTEGER;

    if (upper (:object_type) = 'TABLE') then

        if (position ('.', :object_name) > 0) then
            SIGNAL 'ST100', 'Table name cannot be qualified with database name.';
        end if -- end of check for dot in the name

        SET :stmt_psp_rename = 'ALTER TABLE RENAME ' + :object_name + ' TO ' + :new_name;

    else
        if (upper (:object_type) = 'VIEW') then

            if (position ('.', :object_name) > 0) then
                SIGNAL 'ST100', 'View name cannot be qualified with database name.';
            end if -- end of check for dot in the name

            SET :stmt_psp_rename = 'ALTER VIEW RENAME ' + :object_name + ' TO ' + :new_name;

        else
            if (upper (:object_type) = 'PROCEDURE') then

                if (position ('.', :object_name) > 0) then
                    SIGNAL 'ST100', 'Procedure name cannot be qualified with database name.';
                end if -- end of check for dot in the name

                SET :stmt_psp_rename = 'ALTER PROCEDURE RENAME ' + :object_name + ' TO ' + :new_name;

            else
                if (upper (:object_type) = 'FUNCTION') then

                    if (position ('.', :object_name) > 0) then
                        SIGNAL 'ST100', 'Function name cannot be qualified with database name.';
                    end if -- end of check for dot in the name

                    SET :stmt_psp_rename = 'ALTER FUNCTION RENAME ' + :object_name + ' TO ' + :new_name;

                else
                    if (upper (:object_type) = 'INDEX') then

                        SET :dot_position = position ('.', :object_name);

                        if (:dot_position > 0) then
                            SET :table_name = substring (:object_name, 1, :dot_position - 1);
                            SET :column_name = substring (:object_name, :dot_position + 1,
                                                          length (:object_name) - :dot_position);
                        else
                            SIGNAL 'ST100', 'Table name qualifier required for index renaming.';
                        end if

                        SET :stmt_psp_rename = 'ALTER INDEX RENAME ' + :object_name + ' TO ' + :new_name;

                    else

                        if (upper (:object_type) = 'TRIGGER') then

                            SET :dot_position = position ('.', :object_name);

                            if (:dot_position > 0) then
                                SET :table_name = substring (:object_name, 1, :dot_position - 1);
                                SET :column_name = substring (:object_name, :dot_position + 1,
                                                              length (:object_name) - :dot_position);
                            else
                                SIGNAL 'ST100', 'Table name qualifier required for trigger renaming.';
                            end if

                            SET :stmt_psp_rename = 'ALTER TRIGGER RENAME ' + :object_name + ' TO ' + :new_name;

                        else

                            if (upper (:object_type) = 'COLUMN') then

                                -- find the position of the dot in the table qualified name
                                SET :dot_position = position ('.', :object_name);

                                if (:dot_position > 0) then
                                    SET :table_name = substring (:object_name, 1, :dot_position - 1);
                                    SET :column_name = substring (:object_name, :dot_position + 1,
                                                                  length (:object_name) - :dot_position);
                                else
                                    SIGNAL 'ST100', 'Table name qualifier required for column renaming.';
                                end if

                                SET :stmt_psp_rename = 'ALTER TABLE ' + :table_name + 
                                                       ' RENAME COLUMN ' + :column_name + ' TO ' + :new_name;

                            else
                                SIGNAL 'ST100', 'Invalid object type. Type must be one of the following: Column, Function, Index, Procedure, Table, Trigger, or View.';
                            end if -- end if COLUMN object type comparision

                        end if -- end if TRIGGER object type comparision

                    end if -- end if INDEX object type comparision

                end if -- end if FUNCTION object type comparision

            end if -- end if PROCEDURE object type comparision

        end if -- end if VIEW type comparision

    end if -- end if TABLE type comparision

  execute (:stmt_psp_rename);

END; -- End of stored procedure.

